// Store original methods
const originalXHR = window.XMLHttpRequest;
const originalFetch = window.fetch;

// Create a custom event to communicate with content script
function sendRequestData(details) {
  window.dispatchEvent(new CustomEvent('__xhr_intercepted__', {
    detail: details
  }));
}

// Intercept XHR
class InterceptedXHR extends XMLHttpRequest {
  constructor() {
    super();
    
    this.addEventListener('load', () => {
      sendRequestData({
        type: 'XMLHttpRequest',
        method: this._method,
        url: this._url,
        requestBody: this._requestBody,
        responseBody: this.responseText,
        status: this.status,
        timestamp: Date.now()
      });
    });
  }

  open(method, url) {
    this._method = method;
    this._url = url;
    super.open.apply(this, arguments);
  }

  send(body) {
    this._requestBody = body;
    super.send.apply(this, arguments);
  }
}

// Replace XHR
window.XMLHttpRequest = InterceptedXHR;

// Replace fetch
window.fetch = async (...args) => {
  const request = args[0];
  const options = args[1] || {};
  
  let url = (typeof request === 'string') ? request : request.url;
  let method = (typeof request === 'string') ? (options.method || 'GET') : request.method || 'GET';
  
  // Clone the request body if it exists
  let requestBody = null;
  if (options.body) {
    requestBody = options.body;
  } else if (request instanceof Request && request.body) {
    const clonedRequest = request.clone();
    try {
      requestBody = await clonedRequest.text();
    } catch (e) {
      requestBody = 'Unable to read request body';
    }
  }

  try {
    const response = await originalFetch.apply(window, args);
    const clonedResponse = response.clone();
    const responseBody = await clonedResponse.text();
    
    sendRequestData({
      type: 'fetch',
      method: method,
      url: url,
      requestBody: requestBody,
      responseBody: responseBody,
      status: response.status,
      timestamp: Date.now()
    });

    return response;
  } catch (error) {
    sendRequestData({
      type: 'fetch',
      method: method,
      url: url,
      requestBody: requestBody,
      responseBody: `Error: ${error.message}`,
      status: 0,
      timestamp: Date.now()
    });
    throw error;
  }
};